//
//  SPCSceneExtensions.swift
//
//  Copyright © 2020 Apple Inc. All rights reserved.
//

import SpriteKit
import SPCScene

public extension Point {
    static var center: Point {
        return Point(x: 0.0, y: 0.0)
    }
}

public extension Graphic {
    func animate(with images: [Image], fps: TimeInterval = 10.0, loop: Bool = false) {
        let a = animation(with: images, fps: fps, loop: loop)
        self.run(a)
    }
    
    func animation(with images: [Image], fps: TimeInterval = 10.0, loop: Bool = false) -> SKAction {
        let textures = images.map {
            return SKTexture(image: $0.uiImage)
        }
        let action = SKAction.animate(with: textures, timePerFrame: 1 / fps)
        
        guard loop else {
            return action
        }
        
        return SKAction.repeatForever(action)
    }
}

public extension Sprite {
    override var lightingBitMask: UInt32 {
        didSet {
            let tiles: [SKTileMapNode] = children.compactMap { child in
                guard child is SKTileMapNode else { return nil }
                return child as! SKTileMapNode
            }
            tiles.forEach { $0.lightingBitMask = self.lightingBitMask }
        }
    }
}

public extension Scene {
    func clearLights() {
        let lights: [SKLightNode] = children.compactMap {
            guard $0 is SKLightNode else { return nil }
            return $0 as! SKLightNode
        }
        removeChildren(in: lights)
    }
    
    func enableAccessibility(label: String) {
        accessibilityHints = AccessibilityHints(makeAccessibilityElement: true,
                                                usesCustomAccessibilityLabel: true,
                                                accessibilityLabel: label,
                                                selectImmediately: false, needsUpdatedValue: false)
    }
}

public extension SKLightNode {
    static func torch(at point: Point) -> SKLightNode {
        let lightNode = SKLightNode()
        lightNode.position = CGPoint(x: point.x + 512, y: point.y + 512)
        lightNode.categoryBitMask = 0b0001
        lightNode.lightColor = .white
        lightNode.ambientColor = .black
        lightNode.shadowColor = .black
        lightNode.falloff = 2.0
        return lightNode
    }
}

public extension BaseGraphic {
    func enableAccessibility(label: String, select: Bool = false) {
        accessibilityHints = AccessibilityHints(makeAccessibilityElement: true,
                                                usesCustomAccessibilityLabel: true,
                                                accessibilityLabel: label,
                                                selectImmediately: select, needsUpdatedValue: select)
    }
}

public extension Text {
    static func conversation(_ text: String) -> Text {
        let label = Text(text, size: Size(width: 906, height: 220))
        label.fontName = "Songkhla-Regular"
        label.fontSize = 32
        label.textColor = #colorLiteral(red: 0.0, green: 0.0, blue: 0.0, alpha: 1.0)
        return label
    }
    
    static func artifact(_ text: String) -> Text {
        let label = Text(text, size: Size(width: 580, height: 400))
        label.font = .Menlo
        label.fontSize = 64
        label.textColor = #colorLiteral(red: 0.0, green: 0.0, blue: 0.0, alpha: 1.0)
        label.alignment = .left
        return label
    }
}
